<?php
namespace BLMKT_Connector\App\Controllers;

use BLMKT_Connector as BLMKT_Connector;
use BLMKT_Connector\Core\Controller;
use BLMKT_Connector\App\Models\Settings as Settings_Model;

require_once BLMKT_CONNECTOR_ROOT.'/api/MarketApiConnect.php';

if ( ! class_exists( __NAMESPACE__ . '\\' . 'Global_Actions' ) ) {
	/**
	 * Global Actions
	 *
	 * @since      1.0.0
	 * @package    BLMKT_Connector
	 * @subpackage BLMKT_Connector/controllers
	 */
    class Global_Actions extends Controller
    {

        public function __construct()
        {

        }

        /**
         * Helper: Get post backlink status with "active" as default.
         * This allows lazy-loading without pre-populating all meta on activation.
         *
         * Prevents timeout issues on sites with thousands of posts by not running
         * mass meta updates during plugin activation.
         *
         * @param int $post_id
         * @return string Either 'active' or 'inactive'
         *
         * @since 1.0.0
         */
        private function get_post_backlink_status($post_id) {
            $status = get_post_meta($post_id, '_blmkt_conn_enable_add_backlink', true);
            // Default to 'active' if meta doesn't exist (lazy-loading)
            return empty($status) ? 'active' : $status;
        }

        /**
         * Clear all caches for a specific post.
         * Ensures flawless compatibility with major caching & optimization plugins.
         *
         * Supported plugins:
         * - WP Rocket
         * - W3 Total Cache
         * - WP Super Cache
         * - LiteSpeed Cache
         * - WP Fastest Cache
         * - Autoptimize
         *
         * @param int $post_id The post ID to clear cache for
         *
         * @since 1.0.0
         */
        private function clear_post_cache($post_id) {
            // WordPress Core: Clear object cache
            wp_cache_flush();

            // WP Rocket
            if (function_exists('rocket_clean_post')) {
                rocket_clean_post($post_id);
            }

            // W3 Total Cache
            if (function_exists('w3tc_flush_post')) {
                w3tc_flush_post($post_id);
            }

            // WP Super Cache
            if (function_exists('wp_cache_post_change')) {
                wp_cache_post_change($post_id);
            }

            // LiteSpeed Cache
            if (class_exists('LiteSpeed_Cache_API') && method_exists('LiteSpeed_Cache_API', 'purge_post')) {
                \LiteSpeed_Cache_API::purge_post($post_id);
            }

            // WP Fastest Cache
            if (isset($GLOBALS['wp_fastest_cache']) && method_exists($GLOBALS['wp_fastest_cache'], 'singleDeleteCache')) {
                $GLOBALS['wp_fastest_cache']->singleDeleteCache(false, $post_id);
            }

            // Autoptimize (clear page cache if exists)
            if (class_exists('autoptimizeCache')) {
                \autoptimizeCache::clearall();
            }

            // Allow other plugins to hook into cache clearing
            do_action('blmkt_connector_clear_cache', $post_id);
        }

        /**
         * Register callbacks for actions and filters
         *
         * @since    1.0.0
         */
        public function register_hook_callbacks()
        {
            add_action('save_post', array($this, 'blmkt_check_anchors_on_save'), 10, 3);
            add_action('admin_enqueue_scripts', array($this, 'blmkt_enqueue_scripts'), 100);
            add_action('save_post', array($this, 'blmkt_update_custom_field_blmkt_conn_enable_add_backlink'));
            add_action('trash_post', array($this, 'blmkt_deactivate_page_in_marketplace'));
            add_action('trash_page', array($this, 'blmkt_deactivate_page_in_marketplace'));
            add_action('publish_post', array($this, 'blmkt_add_to_marketplace'));
            add_action('publish_page', array($this, 'blmkt_add_to_marketplace'));
            add_filter('the_content', array($this, 'blmkt_add_ordered_backlink'));
            add_action('admin_notices', array($this, 'blmkt_admin_notice_on_pending_verification'));
            add_action('activated_plugin', array($this, 'blmkt_activation_error'));

            // Register shortcode
            add_shortcode('blmkt-link', array($this, 'blmkt_render_shortcode'));


            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'version-update-request', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'version_update_request'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });
            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'version', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'blmkt_version'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });

            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'get-posts-pages', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'blmkt_getPagesPosts'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });
            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'get-urls', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'blmkt_getPagesPostsUrls'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });
            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'verify-domain', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'blmkt_verify_auth_key'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });
            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'update-key', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'blmkt_update_auth_key'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });
            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'update-posts-pages-orders', array(
                    'methods' => 'POST',
                    'callback' => array($this, 'blmkt_update_posts_pages_orders'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });
            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'get-page-content', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'blmkt_get_single_page_content'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });
            add_action('rest_api_init', function () {
                register_rest_route('wp/v2', 'first-sync', array(
                    'methods' => 'GET',
                    'callback' => array($this, 'blmkt_first_sync'),
                    'permission_callback' => function () {
                        return $this->blmkt_check_authkey() && $this->blmkt_check_ip();
                    }
                ));
            });

        }

     function version_update_request()
        {
            $apiMarket = new \MarketApiConnect(BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY));
            return $apiMarket->updatePluginVersionToMarket();
        }

        function blmkt_version()
        {
            return BLMKT_CONNECTOR_VERSION;
        }

        function blmkt_activation_error()
        {
            file_put_contents(plugin_dir_path(__FILE__) . '/error_activation.html', ob_get_contents());
        }

        function blmkt_check_authkey()
        {
            // Try multiple methods to get the auth header (works across different server configs)
            $key = '';

            // Method 1: getallheaders() if available
            if (function_exists('getallheaders')) {
                $headers = getallheaders();
                if (is_array($headers)) {
                    foreach ($headers as $header => $value) {
                        if (strtolower($header) === 'siteauthkey') {
                            $key = $value;
                            break;
                        }
                    }
                }
            }

            // Method 2: Check $_SERVER variables (works everywhere)
            if (empty($key)) {
                $key = isset($_SERVER['HTTP_SITEAUTHKEY']) ? $_SERVER['HTTP_SITEAUTHKEY'] : '';
            }

            if (strlen($key) > 1 && $key == Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY)) {
                return true;
            }

            return false;
        }

        function blmkt_check_ip()
        {
            $allowedIp = \BLMKT_Connector::get_allowed_ip();
            $testIP = '';

            if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
                $parts = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
                $testIP = trim($parts[0]);
            }

            if ($testIP === '127.0.0.1' || !filter_var($testIP, FILTER_VALIDATE_IP)) {
                $testIP = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
            }

            return filter_var($testIP, FILTER_VALIDATE_IP)
                    && filter_var($allowedIp, FILTER_VALIDATE_IP)
                    && $allowedIp === $testIP;
        }

        public function blmkt_check_anchors_on_save($post_id, $post, $update)
        {
            // Nur bei Updates und veröffentlichten Posts/Pages
            if (!$update || $post->post_status != 'publish') {
                return;
            }

            // Autosaves ignorieren
            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
                return;
            }

            // Prüfen ob Backlink-Bestellungen vorhanden sind
            $backlink_orders = get_post_meta($post_id, '_backlink_orders', true);
            if (empty($backlink_orders)) {
                return;
            }

            $orders = json_decode($backlink_orders);
            if (!is_array($orders)) {
                return;
            }

            // Cooldown prüfen (6 Stunden) um Spam bei häufigem Speichern zu vermeiden
            $last_ping = get_post_meta($post_id, '_blmkt_last_content_sync_ping', true);
            if ($last_ping && (time() - $last_ping) < 21600) {
                return;
            }

            $anchor_missing = false;
            foreach ($orders as $order) {
                if (!empty($order->insertion_anchor)) {
                    if (strpos($post->post_content, $order->insertion_anchor) === false) {
                        $anchor_missing = true;
                        break;
                    }
                }
            }

            // Wenn ein Anker fehlt, Marktplatz benachrichtigen
            if ($anchor_missing) {
                $this->blmkt_ping_marketplace_for_sync($post_id);
                update_post_meta($post_id, '_blmkt_last_content_sync_ping', time());
            }
        }

        private function blmkt_ping_marketplace_for_sync($post_id)
        {
            $apiMarket = new \MarketApiConnect(\BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY));
            
            // Wir nutzen die bestehende Guzzle-Logik im MarketApiConnect (muss dort ergänzt werden)
            // oder senden direkt einen POST Request.
            $serviceUrl = \BLMKT_Connector::BACKLINK_MARKET_URL . "/api/v1/report-content-change";
            
            $postData = json_encode([
                'page_id' => $post_id,
                'site_url' => get_option("siteurl"),
                'auth_key' => Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY)
            ]);

            // Wir nutzen nun die SSL-Logik aus MarketApiConnect
            $ssl_status = get_option('blmkt_ssl_working_status', 'unknown');
            $last_check = get_option('blmkt_ssl_last_check', 0);
            $fail_count = (int) get_option('blmkt_ssl_fail_count', 0);
            $ssl_verify = true;

            // Wenn die Marketplace URL selbst kein HTTPS nutzt
            if (strpos(\BLMKT_Connector::BACKLINK_MARKET_URL, 'https://') === false) {
                $ssl_verify = false;
            } elseif ($ssl_status === 'failing') {
                $delay = 86400 * min($fail_count, 2);
                if ($fail_count >= 3) {
                    $delay = 604800; // 7 Tage
                }
                if ((time() - $last_check) <= $delay) {
                    $ssl_verify = false;
                }
            }

            // Direkter Aufruf via WordPress HTTP API für den Ping
            wp_remote_post($serviceUrl, [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'siteauthkey' => Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY)
                ],
                'body' => $postData,
                'timeout' => 5,
                'sslverify' => apply_filters('blmkt_ssl_verify', $ssl_verify),
                'blocking' => false // Wir warten nicht auf Antwort um den Editor nicht zu bremsen
            ]);
        }

        public function blmkt_getPagesPostsUrls($request_data)
        {
            $args = array(
                'numberposts' => -1,
                'post_status' => array('publish'),
                'meta_key' => '_blmkt_conn_enable_add_backlink',
                'meta_value' => 'active'
            );
            $response = array();
            $all_posts = get_posts($args);
            $i = 0;
            foreach ($all_posts as $all_post) {
                $response[] = get_permalink($all_post->ID);
            }

            $args = array(
                'number' => 0,
                'post_status' => array('publish')
            );
            $all_pages = get_pages($args);
            foreach ($all_pages as $all_page) {
                $response[] = get_permalink($all_page->ID);
            }
            return $response;
        }

        public function blmkt_getPagesPosts($request_data)
        {
            $args = array(
                'numberposts' => -1,
                'post_status' => array('publish'),
                'meta_key' => '_blmkt_conn_enable_add_backlink',
                'meta_value' => 'active'
            );
            $response = array();
            $all_posts = get_posts($args);
            $i = 0;
            foreach ($all_posts as $all_post) {
                $permalink = get_permalink($all_post->ID);
                if (rtrim($permalink, '/') === rtrim(home_url('/'), '/')) {
                    continue; // skip root domain
                }
                $response[$i]['ID'] = $all_post->ID;
                $response[$i]['url'] = get_permalink($all_post->ID);
                $response[$i]['type'] = "post";
                $response[$i]['title'] = get_the_title($all_post->ID);
                $response[$i]['publish_date'] = get_the_date('Y-m-d', $all_post->ID);
                $response[$i]['tags'] = get_the_tags($all_post->ID);
                $post_content = get_post_field('post_content', $all_post->ID);
                $response[$i]['content'] = $post_content;
                $i++;
            }

                  //Get all pages
                  $args = array(
                      'number' => 0,
                      'post_status' => array('publish')
                  );
                  $all_pages = get_pages($args);
                  foreach ($all_pages as $all_page) {
                      $permalink = get_permalink($all_page->ID);
                      if (rtrim($permalink, '/') === rtrim(home_url('/'), '/')) {
                          continue; // skip root domain
                      }
                          $response[$i]['ID'] = $all_page->ID;
                          $response[$i]['type'] = "page";
                          $response[$i]['title'] = get_the_title($all_page->ID);
                          $response[$i]['url'] = get_permalink($all_page->ID);
                          $response[$i]['publish_date'] = get_the_date('Y-m-d', $all_page->ID);
                          $response[$i]['tags'] = get_the_tags($all_page->ID);
                          $post_content = get_post_field('post_content', $all_page->ID);
                      $response[$i]['content'] = $post_content;
                      $i++;
                  }
			return $response;
		}



  public function blmkt_get_single_page_content($request_data)
  {
      $page_id = $request_data->get_param('page_id');
      if (!$page_id) {
          return new \WP_Error('missing_id', 'Page ID is required', array('status' => 400));
      }

      $post = get_post($page_id);
      if (!$post) {
          return new \WP_Error('not_found', 'Page not found', array('status' => 404));
      }

      return array(
          'ID' => $post->ID,
          'content' => $post->post_content,
          'title' => $post->post_title,
          'url' => get_permalink($post->ID)
      );
  }

  public function blmkt_admin_notice_on_pending_verification(){
			if(empty(Settings_Model::get_setting( Settings_Model::BACKLINK_AUTH_KEY )) || Settings_Model::get_setting( Settings_Model::BACKLINK_AUTH_KEY )=="") {
                ?>
                <div class="notice notice-success is-dismissible">
                    <h2>Backlink Market Connector - Confirm your AuthKey</h2>
                    <p><?php _e('To start earning money with your blog, please add the Backlink Market AuthKey. You can find your key <a href="' . BLMKT_Connector::BACKLINK_MARKET_URL . '/seller_sites?confirm=' . urlencode(trailingslashit(get_option("siteurl"))) . '" target="_blank">here</a>', 'backlink-verification_notice'); ?></p>
                </div>
                <?php
            }
		}
        public function blmkt_update_auth_key($request_data){
            $parameters = $request_data->get_params();
			$response=array();
            if( strlen($parameters['new-auth-key'])>5){
                $apiMarket  = new \MarketApiConnect( BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting( Settings_Model::BACKLINK_AUTH_KEY ) );
                if ($apiMarket->updatePluginVersionToMarket() == true) {
                    Settings_Model::update_settings(array("blmkt_connector_auth_key" => $parameters['new-auth-key']));
                    $response['status'] = "success";
                } else {
                    $response['status'] = "forbid";
                }

            }else {
                 $response['status']="fail";
            }
			return $response;
        }

        public function blmkt_update_posts_pages_orders($request_data)
        {
            $parameters = $request_data->get_params();
            $response = array();
            
            // Decode backlink orders
            $backlink_orders = json_decode($parameters['content']);
            if (!is_array($backlink_orders)) {
                return ['status' => 'error', 'msg' => 'Invalid content format'];
            }

            $post = get_post($parameters['content_id']);
            if (!$post) {
                return ['status' => 'error', 'msg' => 'Post not found'];
            }

            $content = $post->post_content;
            $modified = false;

            // Process each order
            foreach ($backlink_orders as $order) {
                if (empty($order->id)) {
                    continue; // Skip if no order ID
                }

                // Create shortcode
                $shortcode = '[blmkt-link id="' . $order->id . '"]';

                // Remove ALL old shortcodes for this link ID completely
                $content = preg_replace('/\[blmkt-link\s+id=["\']?' . preg_quote($order->id, '/') . '["\']?\]/i', '', $content);

                // Check if insertion_anchor exists in content
                if (!empty($order->insertion_anchor)) {
                    // Ensure proper UTF-8 encoding - decode HTML entities first
                    $insertion_anchor = html_entity_decode($order->insertion_anchor, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                    $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');

                    // Use mb_strpos for proper multi-byte string position
                    $anchor_pos = mb_strpos($content, $insertion_anchor);
                    if ($anchor_pos === false) {
                        return ['status' => 'anchor_not_found', 'msg' => "Anchor sentence not found in content"];
                    }

                    // Insert ONE new shortcode after the insertion_anchor - seamless with space
                    $insert_pos = $anchor_pos + mb_strlen($insertion_anchor);
                    $content = mb_substr($content, 0, $insert_pos) . " " . $shortcode . " " . mb_substr($content, $insert_pos);
                    $modified = true;
                }

                // Add embedded_at timestamp to order data
                if (!isset($order->embedded_at)) {
                    $order->embedded_at = current_time('mysql');
                }
            }

            // Update post content if modified
            if ($modified) {
                // Ensure content is properly encoded as UTF-8 for WordPress
                $content = mb_convert_encoding($content, 'UTF-8', 'UTF-8');

                wp_update_post([
                    'ID' => $parameters['content_id'],
                    'post_content' => $content
                ]);

                // Clear caches for this post (flawless compatibility with caching plugins)
                $this->clear_post_cache($parameters['content_id']);
            }

            // Update post meta with order data
            $res = update_post_meta($parameters['content_id'], '_backlink_orders', json_encode($backlink_orders, JSON_UNESCAPED_UNICODE));

            if ($res !== false || !$modified) {
                $response['status'] = "success";
                $response['order_ids'] = array_map(function($order) { return $order->id; }, $backlink_orders);
            } else {
                $response = ['status' => 'error', 'msg' => "Error updating post meta"];
            }

            return $response;

        }

        public function blmkt_first_sync()
        {
            if (Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY) != "") {
                //get all published posts
                $args = array(
                    'numberposts' => -1,
                    'post_status' => array('publish'),
                    'meta_key' => '_blmkt_conn_enable_add_backlink',
                    'meta_value' => 'active'
                );
                $response = array();
                $all_posts = get_posts($args);
                $i = 0;
                foreach ($all_posts as $all_post) {
                    $permalink = get_permalink($all_post->ID);
                      if (rtrim($permalink, '/') === rtrim(home_url('/'), '/')) {
                          continue; // skip root domain
                      }
                    $response[$i]['ID'] = $all_post->ID;
                    $response[$i]['url'] = get_permalink($all_post->ID);
                    $response[$i]['auth_key'] = Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY);
                    $response[$i]['site_url'] = get_option("siteurl");
                    $response[$i]['is_active'] = 1;
                    $post_content =  get_post_field('post_content', $all_post->ID);
                    $response[$i]['content'] = $post_content;
                    $response[$i]['title'] = get_the_title($all_post->ID);
                    $response[$i]['tags'] = get_the_tags($all_post->ID);
                    $response[$i]['type'] = "post";
                    $response[$i]['publish_date'] = get_the_date('Y-m-d', $all_post->ID);
                    $i++;
                }

                //Get all pages
                $args = array(
                    'number' => 0,
                    'post_status' => array('publish'),
                    'meta_key' => '_blmkt_conn_enable_add_backlink',
                    'meta_value' => 'active'
                );
                $all_pages = get_pages($args);
                foreach ($all_pages as $all_page) {
                    $permalink = get_permalink($all_page->ID);
                      if (rtrim($permalink, '/') === rtrim(home_url('/'), '/')) {
                          continue; // skip root domain
                      }
                    $response[$i]['ID'] = $all_page->ID;
                    $response[$i]['url'] = get_permalink($all_page->ID);
                    $response[$i]['auth_key'] = Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY);
                    $response[$i]['site_url'] = get_option("siteurl");
                    $response[$i]['is_active'] = 1;
                    $post_content = get_post_field('post_content', $all_page->ID);
                    $response[$i]['content'] = $post_content;
                    $response[$i]['title'] = get_the_title($all_page->ID);
                    $response[$i]['tags'] = get_the_tags($all_page->ID);
                    $response[$i]['type'] = "page";
                    $response[$i]['publish_date'] = get_the_date('Y-m-d', $all_page->ID);
                    $i++;
                }
            }
            $apiMarket = new \MarketApiConnect(BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY));
            $result_id = $apiMarket->addProductToMarketBulk($response);

            if ($result_id == 0) {
                $response = ['status' => 'error', 'msg' => "Error adding products to market"];
            }
            return $response;
        }

        public function blmkt_backlink_verify_sync_request($vars)
        {
            $vars[] .= 'backlink_verify_sync';
            return $vars;
        }


        public function blmkt_verify_auth_key($request_data)
        {
            $response = array();

            try {
                $auth_key = Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY);
                $site_url = get_option("siteurl");
                $plugin_version = $this->blmkt_version();

                if (empty($auth_key)) {
                    $response['key'] = '';
                } else {
                    $response['key'] = $auth_key;
                }

                $response['site_url'] = $site_url ?: '';
                $response['plugin_version'] = $plugin_version ?: 'unknown';
            } catch (\Exception $e) {
                // Return safe default on error
                $response['key'] = '';
                $response['site_url'] = '';
                $response['plugin_version'] = 'error';
                $response['error'] = $e->getMessage();
            }

            return $response;
        }

        /* //Filter if settings changed to resync all pages and post
        public function plugin_setting_update($old_value, $new_value ){
            if($new_value['blmkt_connector_auth_key']!="" && $old_value['blmkt_connector_auth_key']!=$new_value['blmkt_connector_auth_key'])
            {
                $this->activate();
            }
        }
        */
        public function blmkt_add_to_marketplace($post_id)
        {
            //check if the post is in the database or not with get_post( $post_id ) == null
            if (get_post($post_id) == null) {
                return $post_id;
            }
            $post_raw_data = file_get_contents('php://input');
            $post_raw_data = @json_decode($post_raw_data, true);
            //$new value is "" when not manually changed. Else it will hold active or inactive when manually changed.
            $is_active = $post_raw_data['meta']['_blmkt_conn_enable_add_backlink'];
            if ($is_active == '' || $is_active == "active") {
                $is_active = 1;
            } elseif ($is_active == "inactive") {
                $is_active = 0;
            }
            $_blmkt_conn_enable_add_backlink = get_post_meta($post_id, '_blmkt_conn_enable_add_backlink', true);
            if ($_blmkt_conn_enable_add_backlink == '') { // when $_blmkt_conn_enable_add_backlink ='' its a brand new post
                // New post so transfer it to backlink market
                $post_raw_data = file_get_contents('php://input');
                $post_raw_data = @json_decode($post_raw_data, true);
                $apiMarket = new \MarketApiConnect(BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY));
                $post_content = get_post_field('post_content', $post_id);
                $result_id = $apiMarket->addProductToMarket($post_id, get_permalink($post_id), $is_active, $post_content, get_the_title($post_id), get_the_tags($post_id), get_post_type($post_id), get_the_date('Y-m-d', $post_id));
            }


        }

		public function blmkt_deactivate_page_in_marketplace($post_id)
        {
            //check if the post is in the database or not with get_post( $post_id ) == null
            if (get_post($post_id) == null) {
                return $post_id;
            }
            // Post is trashed so deactivate on backlink market
            $is_active = 0;
            $apiMarket = new \MarketApiConnect(BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY));
            $result_id = $apiMarket->updateProductStateToMarket($post_id, $is_active);
        }

		public function blmkt_save_post_on_activation( $post_id ) {
			//check if the post is in the database or not with get_post( $post_id ) == null
            if (get_post($post_id) == null) {
                return $post_id;
            }
            // check autosave
            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
                return $post_id;
            }
            $is_passed = false;
            $old_value = get_post_meta($post_id, '_blmkt_conn_enable_add_backlink', true);
            //checking if add to backlink is not inactive
            if ($old_value == "" || ($old_value != "" && $old_value == "active")) {
                $is_passed = true;
                update_post_meta($post_id, '_blmkt_conn_enable_add_backlink', "active");
            }
            ob_start();
            $debug_info = ob_get_clean();
            update_post_meta($post_id, 'debug_info', $debug_info);
				$is_active = ((get_post_meta( $post_id, '_blmkt_conn_enable_add_backlink', true ) == 'active') ? 1 : 0);
				$apiMarket  = new \MarketApiConnect( BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting( Settings_Model::BACKLINK_AUTH_KEY ) );
				$post_content=get_post_field('post_content', $post_id);
				$result_id = $apiMarket->addProductToMarket( $post_id, get_permalink($post_id), $is_active,$post_content,get_the_title($post_id),get_the_tags($post_id),get_post_type($post_id), get_the_date('Y-m-d', $post_id) );



				update_post_meta($post_id, 'debug_result', print_r($result_id, true));

        }

        public function blmkt_update_custom_field_blmkt_conn_enable_add_backlink($post_id)
        {
            //check if the post is in the database or not with get_post( $post_id ) == null
            if (get_post($post_id) == null) {
                return $post_id;
            }
            if (get_post_status($post_id) == 'publish') {
                // check autosave
                if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
                    return $post_id;
                }

                $old_value = get_post_meta($post_id, '_blmkt_conn_enable_add_backlink', true);

                $new_value = '';
                if (!empty($_POST)) {
                    $new_value = $_POST['_blmkt_conn_enable_add_backlink'];
                } else {
                    $post_raw_data = file_get_contents('php://input');
                    $post_raw_data = @json_decode($post_raw_data, true);
                    //$new value is "" when not manually changed. Else it will hold active or inactive when manually changed.
                    $new_value = $post_raw_data['meta']['_blmkt_conn_enable_add_backlink'];
                }

                if ($new_value && $new_value != $old_value) {
                    // value in db and different from new -> update to new value
                    $apiMarket = new \MarketApiConnect(BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY));
                    $result_id = $apiMarket->updateProductStateToMarket($post_id, $new_value);
                    update_post_meta($post_id, '_blmkt_conn_enable_add_backlink', $new_value);
                } else if (empty($new_value) && empty($old_value)) {
                    // New post and select not changed so its default (yes=active)
                    update_post_meta($post_id, '_blmkt_conn_enable_add_backlink', 'active');
                }

            }


		}

        public function blmkt_create_custom_field_to_post($post)
        {
            $post_id = get_the_ID();
            if (get_post_type($post_id) != 'post' && get_post_type($post_id) != 'page') {

            }

            wp_nonce_field('blmkt_conn_nonce_' . $post_id, 'blmkt_conn_nonce');
            $apiMarket = new \MarketApiConnect(BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting(Settings_Model::BACKLINK_AUTH_KEY));
            $result_id = $apiMarket->getProductStatusFromMarket($post_id);
            print_r($result_id);
            exit;
            ?>
            <div class="misc-pub-section misc-pub-section-last">
                <label>
                    <?php _e('Add to Backlink Market', BLMKT_Connector::PLUGIN_ID); ?>
                </label>
                <select name="_blmkt_conn_enable_add_backlink">
                    <option value="active" <?php ($enable_add_backlink != 'inactive') ? 'selected' : '' ?>>Yes</option>
                    <option value="inactive" <?php ($enable_add_backlink == 'inactive') ? 'selected' : '' ?>>No</option>
                </select>
            </div>
            <?php
        }

		/**
		 * Register the JavaScript for the admin area.
		 *
		 * @since    1.0.0
		 */
		public function blmkt_enqueue_scripts() {
			global $post;
			wp_enqueue_script(
				BLMKT_Connector::PLUGIN_ID . '_admin_post-js',
				BLMKT_Connector::get_plugin_url() . 'assets/js/admin/blmkt-connector-post.js',
				array( 'wp-edit-post', 'wp-plugins', 'wp-i18n', 'wp-element' ),
				BLMKT_Connector::PLUGIN_VERSION,
				true
			);

			/*$post_is_enable_add_backlink = 0;
			if( isset( $post->ID ) ) {
				$apiMarket  = new \MarketApiConnect( BLMKT_Connector::BACKLINK_MARKET_URL, Settings_Model::get_setting( Settings_Model::BACKLINK_AUTH_KEY ) );
				$post_is_enable_add_backlink = intval($apiMarket->getProductStatusFromMarket( $post->ID ));

				//$post_is_enable_add_backlink = intval(get_post_meta( $post->ID, '_blmkt_conn_enable_add_backlink', true ));

      }

			wp_localize_script( BLMKT_Connector::PLUGIN_ID . '_admin_post-js', 'blmkt_conn_post_vars', array(
				'post_id'                       => isset( $post->ID ) ? $post->ID : null,
                'post_is_enable_add_backlink'   => $post_is_enable_add_backlink,
            ));
			*/
		}

        public function blmkt_add_ordered_backlink( $content ) {
                global $post;
                $backlink_orders=get_post_meta( $post->ID, '_backlink_orders', true );
                if($backlink_orders && $backlink_orders!=""){
                    $backlink_orders =json_decode($backlink_orders);

                    // Check if shortcodes already exist in content
                    // If yes, they will be rendered by blmkt_render_shortcode()
                    // If no, we don't need to do anything (content should have been updated by blmkt_update_posts_pages_orders)
                    foreach($backlink_orders as $backlink_order) {
                        if (strpos($content, '[blmkt-link id="' . $backlink_order->id . '"]') !== false) {
                            // Shortcodes exist, filter will let them be rendered
                            return $content;
                        }
                    }
                }
        return $content;

		}


        /**
         * Render [blmkt-link id="123"] shortcode
         */
        public function blmkt_render_shortcode($atts)
        {
            global $post;

            // Get attributes
            $atts = shortcode_atts([
                'id' => null
            ], $atts);

            if (empty($atts['id'])) {
                return ''; // No ID provided
            }

            // Load backlink orders from post meta
            $backlink_orders = get_post_meta($post->ID, '_backlink_orders', true);
            if (empty($backlink_orders)) {
                return ''; // No orders found
            }

            // Decode orders
            $orders = json_decode($backlink_orders);
            if (!is_array($orders)) {
                return ''; // Invalid format
            }

            // Find the order with matching ID
            $target_order = null;
            foreach ($orders as $order) {
                if (isset($order->id) && $order->id == $atts['id']) {
                    $target_order = $order;
                    break;
                }
            }

            if (!$target_order) {
                return ''; // Order not found
            }

            // Check if link is active (you can add more checks here)
            // For now we just render it

            // Build the link
            $nofollow = '';
            if (isset($target_order->nofollow) && $target_order->nofollow == 1) {
                $nofollow = ' rel="nofollow"';
            }

            // Replace anchor text with link in the paragraph
            $paragraph = $target_order->link_content;
            $anchor = $target_order->anchor_text;
            $url = $target_order->promoted_url;

            // Use preg_replace to replace first occurrence of anchor text with link
            $linked_paragraph = preg_replace(
                '/' . preg_quote($anchor, '/') . '/i',
                '<a href="' . esc_url($url) . '"' . $nofollow . '>' . esc_html($anchor) . '</a>',
                $paragraph,
                1
            );

            // Return the linked paragraph WITHOUT wrapping in p tag (prevents line breaks in inline content)
            return $linked_paragraph;
        }


    }


}

