<?php
namespace BLMKT_Connector\App\Controllers\Admin;

use BLMKT_Connector\App\Controllers\Admin\Base_Controller;
use BLMKT_Connector as BLMKT_Connector;
use BLMKT_Connector\App\Models\Settings as Settings_Model;

if ( ! class_exists( __NAMESPACE__ . '\\' . 'Admin_Settings' ) ) {

	/**
	 * Controller class that implements Plugin Admin Settings configurations
	 *
	 * @since      1.0.0
	 * @package    BLMKT_Connector
	 * @subpackage BLMKT_Connector/controllers/admin
	 */
	class Admin_Settings extends Base_Controller {

		/**
		 * Holds suffix for dynamic add_action called on settings page.
		 *
		 * @var string
		 * @since 1.0.0
		 */
		private static $hook_suffix = 'toplevel_page_' . BLMKT_Connector::PLUGIN_ID . '-settings';

		/**
		 * Slug of the Settings Page
		 *
		 * @since    1.0.0
		 */
		const SETTINGS_PAGE_SLUG = BLMKT_Connector::PLUGIN_ID.'-settings';

		/**
		 * Capability required to access settings page
		 *
		 * @since 1.0.0
		 */
		const REQUIRED_CAPABILITY = 'read';

		/**
		 * Register callbacks for actions and filters
		 *
		 * @since    1.0.0
		 */
		public function register_hook_callbacks() {
			// Create Menu.
			add_action( 'admin_menu', array( $this, 'blmkt_plugin_menu' ) );

			// Enqueue Styles & Scripts.
			add_action( 'admin_print_scripts-' . static::$hook_suffix, array( $this, 'blmkt_enqueue_scripts' ) );
			add_action( 'admin_print_styles-' . static::$hook_suffix, array( $this, 'blmkt_enqueue_styles' ) );

			// Register Fields.
			add_action( 'load-' . static::$hook_suffix, array( $this, 'blmkt_register_fields' ) );

			// Register Settings.
			add_action( 'admin_init', array( $this->model, 'blmkt_register_settings' ) );

			// Settings Link on Plugin's Page.
			add_filter(
				'plugin_action_links_' . BLMKT_Connector::PLUGIN_ID . '/' . BLMKT_Connector::PLUGIN_ID . '.php',
				array( $this, 'add_plugin_action_links' )
			);
		}

		/**
		 * Create menu for Plugin inside Settings menu
		 *
		 * @since    1.0.0
		 */
		public function blmkt_plugin_menu() {

			add_menu_page(
					__( 'Backlink Market', BLMKT_Connector::PLUGIN_ID ),
					__( 'Backlink Market', BLMKT_Connector::PLUGIN_ID ),
					static::REQUIRED_CAPABILITY,
					static::SETTINGS_PAGE_SLUG,             // Menu URL.
					array($this, 'markup_settings_page'),
					'dashicons-editor-unlink',
					55 );

			// @codingStandardsIgnoreStart.
			/*static::$hook_suffix = add_submenu_page( BLMKT_Connector::PLUGIN_ID,
				__( 'Settings', BLMKT_Connector::PLUGIN_ID ),        // Page Title.
				__( 'Settings', BLMKT_Connector::PLUGIN_ID ),        // Menu Title.
				static::REQUIRED_CAPABILITY,           // Capability.
				static::SETTINGS_PAGE_SLUG,             // Menu URL.
				array( $this, 'markup_settings_page' ), // Callback.
				255
			);*/
			// @codingStandardsIgnoreEnd.
		}

		/**
		 * Register the JavaScript for the admin area.
		 *
		 * @since    1.0.0
		 */
		public function blmkt_enqueue_scripts() {
			wp_enqueue_script(
				BLMKT_Connector::PLUGIN_ID . '_admin-js',
				BLMKT_Connector::get_plugin_url() . 'assets/js/admin/blmkt-connector.js',
				array( 'jquery' ),
				BLMKT_Connector::PLUGIN_VERSION,
				true
			);
		}

		/**
		 * Register the JavaScript for the admin area.
		 *
		 * @since    1.0.0
		 */
		public function blmkt_enqueue_styles() {
			wp_enqueue_style(
				BLMKT_Connector::PLUGIN_ID . '_admin-css',
				BLMKT_Connector::get_plugin_url() . 'assets/css/admin/blmkt-connector.css',
				array(),
				BLMKT_Connector::PLUGIN_VERSION,
				'all'
			);
		}

		/**
		 * Creates the markup for the Settings page
		 *
		 * @since    1.0.0
		 */
		public function markup_settings_page() {
			//print_r(get_current_screen());
			if ( current_user_can( static::REQUIRED_CAPABILITY ) ) {
				$this->view->admin_settings_page(
					array(
						'page_title'    => BLMKT_Connector::PLUGIN_NAME,
            'option_page_name' => static::SETTINGS_PAGE_SLUG,
            'option_group_name' => Settings_Model::SETTINGS_NAME,
					)
				);
			} else {
				wp_die( __( 'Access denied.' ) ); // WPCS: XSS OK.
			}
		}

		/**
		 * Registers settings sections and fields
		 *
		 * @since    1.0.0
		 */
		public function blmkt_register_fields() {

			// Add Settings Page Section.
			add_settings_section(
				'blmkt_connector_section',                    // Section ID.
				__( 'Settings', BLMKT_Connector::PLUGIN_ID ), // Section Title.
				array( $this, 'markup_section_headers' ), // Section Callback.
				static::SETTINGS_PAGE_SLUG                 // Page URL.
			);

			// Add Settings Page Field.
			add_settings_field(
                Settings_Model::BACKLINK_AUTH_KEY,                                // Field ID.
				__( 'Backlink Market AuthKey:', BLMKT_Connector::PLUGIN_ID ), // Field Title.
				array( $this, 'markup_text_fields' ),                    // Field Callback.
				static::SETTINGS_PAGE_SLUG,                          // Page.
				'blmkt_connector_section',                              // Section ID.
				array(                                              // Field args.
					'id'        => Settings_Model::BACKLINK_AUTH_KEY,
					'label_for' => Settings_Model::BACKLINK_AUTH_KEY,
                    'class'     => 'blmkt-conn-config'
				)
			);

			// Add Settings Page Field.
			add_settings_field(
                Settings_Model::BACKLINK_ALLOW_CONFIRMATION_CHECKBOX,                                // Field ID.
				__( 'Allow us to insert links and communicate with Backlink-Market.com.', BLMKT_Connector::PLUGIN_ID ), // Field Title.
				array( $this, 'markup_checkbox_fields' ),                    // Field Callback.
				static::SETTINGS_PAGE_SLUG,                          // Page.
				'blmkt_connector_section',                              // Section ID.
				array(                                              // Field args.
					'id'        => Settings_Model::BACKLINK_ALLOW_CONFIRMATION_CHECKBOX,
					'label_for' => Settings_Model::BACKLINK_ALLOW_CONFIRMATION_CHECKBOX,
                    'class'     => 'blmkt-conn-config'
				)
			);
		}

		/**
		 * Adds the section introduction text to the Settings page
		 *
		 * @param array $section Array containing information Section Id, Section
		 *                       Title & Section Callback.
		 *
		 * @since    1.0.0
		 */
		public function markup_section_headers( $section ) {
			$this->view->section_headers(
				array(
					'section'      => $section,
					'str_header_description' => __( '<p>Please enter the AuthKey you\'ve received from Backlink Market.</p>', BLMKT_Connector::PLUGIN_ID ),
				)
			);
		}

		/**
		 * Delivers the markup for settings fields
		 *
		 * @param array $field_args Field arguments passed in `add_settings_field`
		 *                          function.
		 *
		 * @since    1.0.0
		 */
		public function markup_text_fields( $field_args ) {
			$field_id = $field_args['id'];
			$settings_value = $this->model->get_setting( $field_id );
			$this->view->markup_text_fields(
				array(
					'field_id'       => esc_attr( $field_id ),
          'settings_name'  => $this->get_model()->get_plugin_settings_option_key(),
					'settings_value' => ! empty( $settings_value ) ? esc_attr( $settings_value ) : '',
				)
			);
		}

		/**
		 * Delivers the markup for settings fields
		 *
		 * @param array $field_args Field arguments passed in `add_settings_field`
		 *                          function.
		 *
		 * @since    1.0.0
		 */
		public function markup_checkbox_fields( $field_args ) {
			$field_id = $field_args['id'];
			$settings_value = $this->model->get_setting( $field_id );
			$this->view->markup_checkbox_fields(
				array(
					'field_id'       => esc_attr( $field_id ),
          'settings_name'  => $this->get_model()->get_plugin_settings_option_key(),
					'settings_value' => ! empty( $settings_value ) ? esc_attr( $settings_value ) : '',
				)
			);
		}

    /**
     * Delivers the markup for settings fields
     *
     * @param array $field_args Field arguments passed in `add_settings_field`
     *                          function.
     *
     * @since    1.0.0
     */
    public function markup_html_fields( $field_args ) {
        $field_id = $field_args['id'];
        $settings_value = $this->model->get_setting( $field_id );
        $this->view->markup_html_fields(
            array(
                'field_id'       => esc_attr( $field_id ),
                'settings_name'  => $this->get_model()->get_plugin_settings_option_key(),
                'settings_value' => ! empty( $settings_value ) ? esc_attr( $settings_value ) : '',
            )
        );
    }

		/**
		 * Adds links to the plugin's action link section on the Plugins page
		 *
		 * @param array $links The links currently mapped to the plugin.
		 * @return array
		 *
		 * @since    1.0.0
		 */
		public function add_plugin_action_links( $links ) {
			$settings_link = '<a href="' . admin_url( 'admin.php?page=' . static::SETTINGS_PAGE_SLUG ) . '">' . __( 'Settings', BLMKT_Connector::PLUGIN_ID ) . '</a>';
			array_unshift( $links, $settings_link );

			return $links;
		}

	}

}
