<?php
namespace BLMKT_Connector\App;

/**
 * Fired during plugin uninstallation.
 *
 * This class defines all code necessary to run during the plugin's uninstallation.
 *
 * @since      1.0.0
 * @package    BLMKT_Connector
 * @subpackage BLMKT_Connector/App
 */
class Uninstaller {

    const MU_PLUGIN_FILE = 'blmkt-shortcode-hider.php';

    /**
     * Run uninstall procedure.
     *
     * @since 1.0.0
     */
    public function uninstall() {
        global $wpdb;

        // Step 1: Delete order table
        $table_name = $wpdb->prefix . 'blmkt_connector_orders';
        $wpdb->query("DROP TABLE IF EXISTS $table_name");

        // Step 2: Delete all post meta keys in ONE query (optimized)
        $meta_keys = array(
            '_blmkt_conn_enable_add_backlink',  // Plugin enable/disable flag
            '_backlink_orders',                 // Backlink orders data
            'debug_result',                     // Debug result info
            'debug_info'                        // Debug info
        );

        // Use IN statement for single optimized query
        $placeholders = implode(',', array_fill(0, count($meta_keys), '%s'));
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->postmeta} WHERE meta_key IN ($placeholders)",
                $meta_keys
            )
        );

        // Step 3: Create MU-Plugin to hide shortcodes
        $this->create_shortcode_hider();
    }

    /**
     * Create MU-Plugin that hides [blmkt-link] shortcodes.
     * This is safer than deleting them from the database.
     *
     * @since 1.0.0
     */
    private function create_shortcode_hider() {
        $mu_plugins_dir = WP_CONTENT_DIR . '/mu-plugins';

        // Create directory if it doesn't exist
        if (!file_exists($mu_plugins_dir)) {
            @wp_mkdir_p($mu_plugins_dir);
        }

        $file_path = $mu_plugins_dir . '/' . self::MU_PLUGIN_FILE;

        // Only create if not already exists
        if (!file_exists($file_path)) {
            $code = '<?php
/**
 * BLMKT Shortcode Hider
 * 
 * Hides [blmkt-link] shortcodes after BLMKT Connector uninstall.
 * This prevents broken shortcodes from appearing on the frontend.
 * 
 * @since 1.0.0
 */

add_shortcode(\'blmkt-link\', function() {
    return \'\'; // Return empty string to hide shortcode
});
';

            @file_put_contents($file_path, $code);
        }
    }
}
