<?php

require_once __DIR__.'/MarketApiException.php';
require_once __DIR__.'/MarketApiBasic.php';

/**
 * contains function for connecting against blmkt-connect api
 * @author Patric Eid
 *
 */
class MarketApiConnect extends MarketApiBasic {
    private $api_url_product = "/api/v1/products";
    private $api_url_product_bulk = "/api/v1/products-bulk";
    private $api_url_change_product_state = "/api/v1/products/change-state";
    private $api_url_get_product_status = "/api/v1/products/product-status";
    private $api_url_get_ordered_backlink = "/api/v1/products/check-ordered";
    private $api_url_get_key_update_status = "/api/v1/site/check-key-update-status";
    private $api_url_update_plugin_version = "/api/v1/site/update-plugin-version";
    private $api_wp_site_url = "";
    private $api_auth_key = "";
	private $debugMode = false;

	function __construct($api_wp_site_url, $api_auth_key) {
	    $this->api_wp_site_url = $api_wp_site_url;
	    $this->api_auth_key = $api_auth_key;
	}

	/**
	 * Returns whether SSL verification should be enabled.
	 * 
	 * @return bool
	 */
	private function ssl_verify() {
		// Manual override via filter still has highest priority
		$filtered = apply_filters('blmkt_ssl_verify', null);
		if ($filtered !== null) {
			return (bool) $filtered;
		}

		// If the market URL itself is not HTTPS, we don't need SSL verification
		if (strpos($this->api_wp_site_url, 'https://') === false) {
			return false;
		}

		// Check if we have a stored failure status
		$ssl_status = get_option('blmkt_ssl_working_status', 'unknown');
		$last_check = get_option('blmkt_ssl_last_check', 0);
		$fail_count = (int) get_option('blmkt_ssl_fail_count', 0);

		// If SSL is marked as failing, use exponential backoff for retries
		if ($ssl_status === 'failing') {
			// Base delay is 24 hours, increases with each failure up to 1 week
			// 1 failure: 24h, 2: 48h, 3+: 1 week (604800 seconds)
			$delay = 86400 * min($fail_count, 2);
			if ($fail_count >= 3) {
				$delay = 604800; // 7 days
			}

			if ((time() - $last_check) > $delay) {
				return true; // Try with SSL again after the delay
			}
			return false; // Stay on non-SSL for now
		}

		return true; // Default to secure
	}


	/**
	 * Updates the SSL status based on request results.
	 * 
	 * @param mixed $response
	 * @param bool $was_ssl_attempt
	 */
	private function update_ssl_status($response, $was_ssl_attempt) {
		if ($was_ssl_attempt) {
			if (!is_wp_error($response)) {
				update_option('blmkt_ssl_working_status', 'working');
				update_option('blmkt_ssl_last_check', time());
				update_option('blmkt_ssl_fail_count', 0);
			} else {
				$error_message = $response->get_error_message();
				// Check if it's an SSL related error
				if (strpos($error_message, 'SSL') !== false || strpos($error_message, 'certificate') !== false || strpos($error_message, 'curl error 60') !== false || strpos($error_message, 'curl error 77') !== false) {
					$fail_count = (int) get_option('blmkt_ssl_fail_count', 0) + 1;
					update_option('blmkt_ssl_working_status', 'failing');
					update_option('blmkt_ssl_last_check', time());
					update_option('blmkt_ssl_fail_count', $fail_count);
				}
			}
		}
	}

	/**
     * executes a get call
     * @param string $serviceUrl
     * @return string
     */
	private function _GET($serviceUrl) {
		$ssl_verify = $this->ssl_verify();
		$args = array(
			'headers'=>$this->getHeader(),
			'method'=>'GET',
			'timeout'=>60,
			'sslverify' => $ssl_verify
		);

	    $response = wp_remote_get($serviceUrl, $args);

		// If it failed and we were trying with SSL, try one more time without SSL
		if (is_wp_error($response) && $ssl_verify === true) {
			$this->update_ssl_status($response, true);
			$args['sslverify'] = false;
			$response = wp_remote_get($serviceUrl, $args);
		} else {
			$this->update_ssl_status($response, $ssl_verify);
		}

	    return $this->setReturn($response);
    }

    /**
     * executes a POST call
     * @param string $serviceUrl
     * @param string $postData
     * @return string
     */
	private function _POST($serviceUrl, $postData) {
		$ssl_verify = $this->ssl_verify();
		$args = array(
			'headers'=>$this->getHeader(),
			'body' => $postData,
			'method'=>'POST',
			'timeout'=>60,
			'sslverify' => $ssl_verify
		);

	    $response = wp_remote_post($serviceUrl, $args);

		if (is_wp_error($response) && $ssl_verify === true) {
			$this->update_ssl_status($response, true);
			$args['sslverify'] = false;
			$response = wp_remote_post($serviceUrl, $args);
		} else {
			$this->update_ssl_status($response, $ssl_verify);
		}

	    return $this->setReturn($response);
	}

    /**
     * executes a delete call
     * @param string $serviceUrl
     * @param string $postData
     * @return string
     */
	private function _DELETE($serviceUrl, $postData) {
		$ssl_verify = $this->ssl_verify();
		$args = array(
			'headers'=>$this->getHeader(),
			'body' => $postData,
			'method'=>'DELETE',
			'timeout'=>60,
			'sslverify' => $ssl_verify
		);

		$response = wp_remote_post($serviceUrl, $args);

		if (is_wp_error($response) && $ssl_verify === true) {
			$this->update_ssl_status($response, true);
			$args['sslverify'] = false;
			$response = wp_remote_post($serviceUrl, $args);
		} else {
			$this->update_ssl_status($response, $ssl_verify);
		}

		return $this->setReturn($response);
	}

    /**
     * executes a put call
     * @param string $serviceUrl
     * @param string $postData
     * @return string
     */
	private function _PUT($serviceUrl, $postData) {
		$ssl_verify = $this->ssl_verify();
		$args = array(
			'headers'=>$this->getHeader(),
			'body' => $postData,
			'method'=>'PUT',
			'timeout'=>60,
			'sslverify' => $ssl_verify
		);

		$response = wp_remote_request($serviceUrl, $args);

		if (is_wp_error($response) && $ssl_verify === true) {
			$this->update_ssl_status($response, true);
			$args['sslverify'] = false;
			$response = wp_remote_request($serviceUrl, $args);
		} else {
			$this->update_ssl_status($response, $ssl_verify);
		}

		return $this->setReturn($response);
	}

	/**
	 * returns headers array
	 * @return array
	 */
	private function getHeader() {
		return array(
            'Content-Type' => 'application/json; charset=UTF-8',
            'siteauthkey' => $this->api_auth_key,
            'siteurl' => get_option("siteurl")
        );
	}

    /**
     * evaluates call response and returns an array with response information
     * @param  $response
     * @return array|stdClass
     */
    private function setReturn($response) {

        if (is_wp_error($response)) {
            return $response->get_error_message();
        }else {
            return $response['body'];
        }
    }

    public function addProductToMarket( $page_id, $page_url, $is_active,$content="",$title="",$tags="",$post_type="",$publish_date="" )
    {
        $postData = array(
            'page_id' => $page_id,
            'page_url' => $page_url,
            'auth_key' => $this->api_auth_key,
            'site_url' => get_option("siteurl"),
            'is_active' => $is_active,
            'content' => $content,
            'title' => $title,
            'tags' => $tags,
            'type' => $post_type,
            'publish_date' => $publish_date
        );
        // Skip if root domain is same as url
        if (rtrim($page_url, '/') === rtrim(get_option("siteurl"), '/')) {
            return 0;
        }
        $postDataStr = json_encode($postData);
        $ret_data = $this->_POST($this->api_wp_site_url . $this->api_url_product, $postDataStr);
        $ret_data = json_decode($ret_data, true);
        if (isset($ret_data['product'])) {
            return $ret_data['product']['id'];
        } else {
            return 0;
        }
    }

    public function addProductToMarketBulk($data)
    {

        $postDataStr = json_encode($data);
        $ret_data = $this->_POST($this->api_wp_site_url . $this->api_url_product_bulk, $postDataStr);
        $ret_data = json_decode($ret_data, true);
        if (isset($ret_data['status']) && $ret_data['status'] == "success") {
            return $ret_data;
        } else {
            return 0;
        }
    }

    public function updateProductStateToMarket($page_id, $is_active)
    {
        $postData = array(
            'page_id' => $page_id,
            'auth_key' => $this->api_auth_key,
            'site_url' => get_option("siteurl"),
            'is_active' => $is_active,
        );
        $postDataStr = json_encode($postData);
        $ret_data = $this->_POST($this->api_wp_site_url . $this->api_url_change_product_state, $postDataStr);
        $ret_data = json_decode($ret_data, true);
        if(isset($ret_data['product'])) {
            return $ret_data['product']['id'];
        } else {
            return 0;
        }
    }

    public function getProductStatusFromMarket($page_id){
      $postData = array(
          'page_id' => $page_id,
          'auth_key' => $this->api_auth_key,
          'site_url' => get_option("siteurl"),
      );
      $postDataStr = json_encode($postData);
      $ret_data = $this->_POST( $this->api_wp_site_url . $this->api_url_get_product_status, $postDataStr);
      $ret_data = json_decode($ret_data, true);
      if(isset($ret_data['product']) && count($ret_data['product'])>0) {
          return $ret_data['product']['is_active'];
      } else {
          return 0;
      }
    }

    public function getBacklinkOrderContent($page_id){
      $postData = array(
          'page_id' => $page_id,
          'auth_key' => $this->api_auth_key,
          'site_url' => get_option("siteurl"),
      );
      $postDataStr = json_encode($postData);
      $ret_data = $this->_POST( $this->api_wp_site_url . $this->api_url_get_ordered_backlink, $postDataStr);
      $ret_data = json_decode($ret_data, true);
      if(isset($ret_data['content'])) {
          return $ret_data['content'];
      } else {
          return "";
      }
    }

    public function getIsUpdateAuthKeyAllowed(){
         $postData = array(
             'auth_key' => $this->api_auth_key,
             'site_url' => get_option("siteurl"),
         );
        $postDataStr = json_encode($postData);
        $ret_data = $this->_POST($this->api_wp_site_url . $this->api_url_get_key_update_status, $postDataStr);
        $ret_data = json_decode($ret_data, true);
        if (isset($ret_data['status']) && $ret_data['status'] == "allowed") {
            return true;
        } else {
            return false;
        }
    }

    public function updatePluginVersionToMarket()
    {
        $postData = array(
            'auth_key' => $this->api_auth_key,
            'site_url' => get_option("siteurl"),
            'plugin_version' => BLMKT_CONNECTOR_VERSION
        );
        $postDataStr = json_encode($postData);
        $ret_data = $this->_POST($this->api_wp_site_url . $this->api_url_update_plugin_version, $postDataStr);
        $ret_data = json_decode($ret_data, true);
        return $ret_data;

    }

}
